<?php
if (!defined('ABSPATH')) exit;

/**
 * Validate and sanitize submitted values based on schema.
 *
 * @return array{ok:bool, values:array, errors:array}
 */
function warforms_reboot_validate(array $schema, array $submitted): array {
    $values = [];
    $errors = [];

    $fields = is_array($schema['fields'] ?? null) ? $schema['fields'] : [];

    // Helper: fetch value by field name
    $get = function(string $name) use ($submitted) {
        return $submitted[$name] ?? null;
    };

    foreach ($fields as $field) {
        if (!is_array($field)) continue;
        $type = sanitize_key((string)($field['type'] ?? 'text'));
        $name = sanitize_key((string)($field['name'] ?? ''));
        $label = (string)($field['label'] ?? $name);
        $required = !empty($field['required']);

        if ($name === '') continue;

        $raw = $get($name);

        // normalize emptiness
        $is_empty = false;
        if (is_array($raw)) {
            $is_empty = (count(array_filter($raw, function($v){ return $v !== '' && $v !== null; })) === 0);
        } else {
            $is_empty = (trim((string)$raw) === '');
        }

        if ($required && $is_empty) {
            $errors[$name] = 'This field is required.';
            continue;
        }

        // If empty and not required, store as empty and move on.
        if ($is_empty) {
            $values[$name] = (in_array($type, ['multiselect','checkbox_group'], true)) ? [] : '';
            continue;
        }

        if ($type === 'email') {
            $v = sanitize_email((string)$raw);
            if ($v === '' || !is_email($v)) {
                $errors[$name] = 'Please enter a valid email address.';
                continue;
            }
            $values[$name] = $v;
            continue;
        }

        if ($type === 'url') {
            $v = esc_url_raw((string)$raw);
            if ($v === '' || !filter_var($v, FILTER_VALIDATE_URL)) {
                $errors[$name] = 'Please enter a valid URL.';
                continue;
            }
            $values[$name] = $v;
            continue;
        }

        if ($type === 'phone') {
            $v_raw = trim(sanitize_text_field((string)$raw));

            if ($v_raw === '' && !$required) {
                $values[$name] = '';
                continue;
            }

            /**
             * Accept common North American formats and normalize to E.164.
             * Examples accepted:
             *  - 5555555555
             *  - (555) 555-5555
             *  - 1-555-555-5555
             *  - +15555555555
             *
             * Stored as: +1########## (or +<country><number> for 11-15 digits).
             */
            $digits = preg_replace('/\D+/', '', $v_raw);

// NANP only: accept 10 digits (auto +1) or 11 digits starting with 1.
$e164 = '';
$len = strlen($digits);

if ($len === 10) {
    $e164 = '+1' . $digits;
} elseif ($len === 11 && isset($digits[0]) && $digits[0] === '1') {
    $e164 = '+' . $digits;
}

/**
 * Filter normalized phone (E.164) before storing.
 * Return empty string to force validation error.
 */
$e164 = apply_filters('warforms_reboot_normalized_phone', $e164, $v_raw, $digits, $name, $field, $schema);

if (!is_string($e164) || $e164 === '' || !preg_match('/^\+1\d{10}$/', $e164)) {
    $errors[$name] = 'Please enter a valid US/Canada phone number (10 digits).';
    continue;
}

$values[$name] = $e164;
continue;
        }

        if ($type === 'date') {
            $v = sanitize_text_field((string)$raw);
            // Expect YYYY-MM-DD (HTML date)
            if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $v)) {
                $errors[$name] = 'Please enter a valid date.';
                continue;
            }
            $values[$name] = $v;
            continue;
        }

        if (in_array($type, ['select','radio'], true)) {
            $opts = is_array($field['options'] ?? null) ? $field['options'] : [];
            $allowed = array_map('strval', $opts);
            $v = sanitize_text_field((string)$raw);
            if ($v === '' && !$required) {
                $values[$name] = '';
                continue;
            }
            if (!in_array($v, $allowed, true)) {
                $errors[$name] = 'Invalid selection.';
                continue;
            }
            $values[$name] = $v;
            continue;
        }

        if (in_array($type, ['multiselect','checkbox_group'], true)) {
            $opts = is_array($field['options'] ?? null) ? $field['options'] : [];
            $allowed = array_map('strval', $opts);

            $arr = is_array($raw) ? $raw : [$raw];
            $clean = [];
            foreach ($arr as $item) {
                $item = sanitize_text_field((string)$item);
                if ($item === '') continue;
                if (!in_array($item, $allowed, true)) {
                    $errors[$name] = 'Invalid selection.';
                    continue 2;
                }
                $clean[] = $item;
            }
            $values[$name] = array_values(array_unique($clean));
            continue;
        }

        if ($type === 'currency') {
            $allowed = is_array($field['allowed'] ?? null) ? $field['allowed'] : ['USD'];
            $allowed = array_values(array_unique(array_map(function($c){
                $c = strtoupper(preg_replace('/[^A-Z]/', '', (string)$c));
                return strlen($c) === 3 ? $c : '';
            }, $allowed)));
            $allowed = array_filter($allowed);

            $curr = '';
            $amt = '';
            if (is_array($raw)) {
                $curr = isset($raw['currency']) ? (string)$raw['currency'] : '';
                $amt = isset($raw['amount']) ? (string)$raw['amount'] : '';
            }
            $curr = strtoupper(preg_replace('/[^A-Z]/', '', $curr));
            $amt  = trim((string)$amt);

            if ($required && ($curr === '' || $amt === '')) {
                $errors[$name] = 'Currency and amount are required.';
                continue;
            }
            if ($curr !== '' && !in_array($curr, $allowed, true)) {
                $errors[$name] = 'Invalid currency.';
                continue;
            }
            if ($amt !== '' && !preg_match('/^-?\d+(\.\d{1,2})?$/', $amt)) {
                $errors[$name] = 'Amount must be a number (up to 2 decimals).';
                continue;
            }

            // Normalize
            if ($amt !== '') {
                $amt = number_format((float)$amt, 2, '.', '');
            }

            $values[$name] = ['currency' => $curr ?: ($allowed[0] ?? 'USD'), 'amount' => $amt];
            continue;
        }

        // default text-like
        if ($type === 'textarea') {
            $values[$name] = sanitize_textarea_field((string)$raw);
        } else {
            $values[$name] = sanitize_text_field((string)$raw);
        }
    }

    return [
        'ok' => empty($errors),
        'values' => $values,
        'errors' => $errors,
    ];
}
